///////////////////////////////////////////////////////////////////////////////////////
//
// MonoScan
//
// 03/05/07		Hounddog	Fixed WindowState bug
// 02/05/07		Hounddog	Fixed OnClosing() bug
// 27/04/07		Hounddog	Initial implementation
//

using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.Text;
using System.Configuration;
using System.Xml;
using System.Reflection;
using System.Runtime.InteropServices;

namespace MonoScan
{
	public class MainForm : System.Windows.Forms.Form
	{
		private System.Windows.Forms.MainMenu mainMenu;
		private System.Windows.Forms.StatusBar statusBar;
		private System.Windows.Forms.MenuItem menuItem9;
		private System.Windows.Forms.MenuItem scannerMenuItem;
		private System.Windows.Forms.MenuItem scannerConnectMenuItem;
		private System.Windows.Forms.MenuItem scannerDisconnectMenuItem;
		private System.Windows.Forms.MenuItem scannerExitMenuItem;
		private System.Windows.Forms.MenuItem helpMenuItem;
		private System.Windows.Forms.MenuItem helpAboutMenuItem;
		private System.Windows.Forms.MenuItem menuItem1;
		private System.Windows.Forms.MenuItem scannerLogMenuItem;
		private System.Windows.Forms.Timer idleTimer;
		private System.Windows.Forms.MenuItem menuItem2;
		private System.Windows.Forms.MenuItem windowMenuItem;
		private System.Windows.Forms.MenuItem windowTileHorizontallyMenuItem;
		private System.Windows.Forms.MenuItem windowTileVerticallyMenuItem;
		private System.Windows.Forms.MenuItem windowCascadeMenuItem;
		private System.Windows.Forms.MenuItem windowCloseAllMenuItem;
		private System.Windows.Forms.MenuItem scannerOptionsMenuItem;
		private MonoScan.LogBox logBox;
		private System.Windows.Forms.Splitter logSplitter;
		private System.ComponentModel.IContainer components;

		public MainForm()
		{
			// Creation and intialization

			InitializeComponent();

			monoCommMessageHandlerDelegate = new MonoComm.MessageHandlerDelegate(this.MonoCommMessageHandler);

			statusMessageArrayList = new ArrayList();

			// XML metadata

			xmlDocument = new XmlDocument();

			xmlDocument.Load(Configuration.Path);
			xmlNode = Configuration.GetXmlNode(xmlDocument, "MonoScan");

			// MonoComm

			MonoComm.Status status = MonoComm.Construct(IntPtr.Zero, monoCommMessageHandlerDelegate);

			if (status != MonoComm.Status.Ok)
				throw new MonoComm.Exception(status);

			constructed = true;

			// Idle timer

			idleTimer.Interval = (int) (MonoComm.GetMaxIdleInterval() / 2);

			// Configuration

			ReadConfiguration();
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.components = new System.ComponentModel.Container();
			this.mainMenu = new System.Windows.Forms.MainMenu();
			this.scannerMenuItem = new System.Windows.Forms.MenuItem();
			this.scannerConnectMenuItem = new System.Windows.Forms.MenuItem();
			this.scannerDisconnectMenuItem = new System.Windows.Forms.MenuItem();
			this.menuItem1 = new System.Windows.Forms.MenuItem();
			this.scannerLogMenuItem = new System.Windows.Forms.MenuItem();
			this.scannerOptionsMenuItem = new System.Windows.Forms.MenuItem();
			this.menuItem9 = new System.Windows.Forms.MenuItem();
			this.scannerExitMenuItem = new System.Windows.Forms.MenuItem();
			this.windowMenuItem = new System.Windows.Forms.MenuItem();
			this.windowTileHorizontallyMenuItem = new System.Windows.Forms.MenuItem();
			this.windowTileVerticallyMenuItem = new System.Windows.Forms.MenuItem();
			this.windowCascadeMenuItem = new System.Windows.Forms.MenuItem();
			this.menuItem2 = new System.Windows.Forms.MenuItem();
			this.windowCloseAllMenuItem = new System.Windows.Forms.MenuItem();
			this.helpMenuItem = new System.Windows.Forms.MenuItem();
			this.helpAboutMenuItem = new System.Windows.Forms.MenuItem();
			this.statusBar = new System.Windows.Forms.StatusBar();
			this.idleTimer = new System.Windows.Forms.Timer(this.components);
			this.logBox = new MonoScan.LogBox();
			this.logSplitter = new System.Windows.Forms.Splitter();
			this.SuspendLayout();
			// 
			// mainMenu
			// 
			this.mainMenu.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																					 this.scannerMenuItem,
																					 this.windowMenuItem,
																					 this.helpMenuItem});
			// 
			// scannerMenuItem
			// 
			this.scannerMenuItem.Index = 0;
			this.scannerMenuItem.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																							this.scannerConnectMenuItem,
																							this.scannerDisconnectMenuItem,
																							this.menuItem1,
																							this.scannerLogMenuItem,
																							this.scannerOptionsMenuItem,
																							this.menuItem9,
																							this.scannerExitMenuItem});
			this.scannerMenuItem.Text = "&Connection";
			this.scannerMenuItem.Popup += new System.EventHandler(this.scannerMenuItem_Popup);
			// 
			// scannerConnectMenuItem
			// 
			this.scannerConnectMenuItem.Index = 0;
			this.scannerConnectMenuItem.Text = "&Connect";
			this.scannerConnectMenuItem.Click += new System.EventHandler(this.scannerConnectMenuItem_Click);
			// 
			// scannerDisconnectMenuItem
			// 
			this.scannerDisconnectMenuItem.Index = 1;
			this.scannerDisconnectMenuItem.Text = "&Disconnect";
			this.scannerDisconnectMenuItem.Click += new System.EventHandler(this.scannerDisconnectMenuItem_Click);
			// 
			// menuItem1
			// 
			this.menuItem1.Index = 2;
			this.menuItem1.Text = "-";
			// 
			// scannerLogMenuItem
			// 
			this.scannerLogMenuItem.Index = 3;
			this.scannerLogMenuItem.Text = "Logging";
			this.scannerLogMenuItem.Click += new System.EventHandler(this.scannerLogMenuItem_Click);
			// 
			// scannerOptionsMenuItem
			// 
			this.scannerOptionsMenuItem.Index = 4;
			this.scannerOptionsMenuItem.Text = "&Settings...";
			this.scannerOptionsMenuItem.Click += new System.EventHandler(this.scannerOptionsMenuItem_Click);
			// 
			// menuItem9
			// 
			this.menuItem9.Index = 5;
			this.menuItem9.Text = "-";
			// 
			// scannerExitMenuItem
			// 
			this.scannerExitMenuItem.Index = 6;
			this.scannerExitMenuItem.Text = "&Exit";
			this.scannerExitMenuItem.Click += new System.EventHandler(this.scannerExitMenuItem_Click);
			// 
			// windowMenuItem
			// 
			this.windowMenuItem.Index = 1;
			this.windowMenuItem.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																						   this.windowTileHorizontallyMenuItem,
																						   this.windowTileVerticallyMenuItem,
																						   this.windowCascadeMenuItem,
																						   this.menuItem2,
																						   this.windowCloseAllMenuItem});
			this.windowMenuItem.Text = "&Window";
			this.windowMenuItem.Popup += new System.EventHandler(this.WindowMenuItem_Popup);
			// 
			// windowTileHorizontallyMenuItem
			// 
			this.windowTileHorizontallyMenuItem.Index = 0;
			this.windowTileHorizontallyMenuItem.Text = "Tile &Horizontally";
			this.windowTileHorizontallyMenuItem.Click += new System.EventHandler(this.windowTileHorizontallyMenuItem_Click);
			// 
			// windowTileVerticallyMenuItem
			// 
			this.windowTileVerticallyMenuItem.Index = 1;
			this.windowTileVerticallyMenuItem.Text = "Tile &Vertically";
			this.windowTileVerticallyMenuItem.Click += new System.EventHandler(this.windowTileVerticallyMenuItem_Click);
			// 
			// windowCascadeMenuItem
			// 
			this.windowCascadeMenuItem.Index = 2;
			this.windowCascadeMenuItem.Text = "&Cascade";
			this.windowCascadeMenuItem.Click += new System.EventHandler(this.windowCascadeMenuItem_Click);
			// 
			// menuItem2
			// 
			this.menuItem2.Index = 3;
			this.menuItem2.Text = "-";
			// 
			// windowCloseAllMenuItem
			// 
			this.windowCloseAllMenuItem.Index = 4;
			this.windowCloseAllMenuItem.Text = "Close &All";
			this.windowCloseAllMenuItem.Click += new System.EventHandler(this.windowCloseAllMenuItem_Click);
			// 
			// helpMenuItem
			// 
			this.helpMenuItem.Index = 2;
			this.helpMenuItem.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																						 this.helpAboutMenuItem});
			this.helpMenuItem.Text = "&Help";
			// 
			// helpAboutMenuItem
			// 
			this.helpAboutMenuItem.Index = 0;
			this.helpAboutMenuItem.Text = "&About...";
			this.helpAboutMenuItem.Click += new System.EventHandler(this.helpAboutMenuItem_Click);
			// 
			// statusBar
			// 
			this.statusBar.Location = new System.Drawing.Point(0, 429);
			this.statusBar.Name = "statusBar";
			this.statusBar.Size = new System.Drawing.Size(752, 20);
			this.statusBar.TabIndex = 0;
			// 
			// idleTimer
			// 
			this.idleTimer.Tick += new System.EventHandler(this.idleTimer_Tick);
			// 
			// logBox
			// 
			this.logBox.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
			this.logBox.Dock = System.Windows.Forms.DockStyle.Bottom;
			this.logBox.Location = new System.Drawing.Point(0, 301);
			this.logBox.Name = "logBox";
			this.logBox.Size = new System.Drawing.Size(752, 128);
			this.logBox.TabIndex = 0;
			this.logBox.TabStop = false;
			this.logBox.TextCapacity = 1048576;
			// 
			// logSplitter
			// 
			this.logSplitter.Dock = System.Windows.Forms.DockStyle.Bottom;
			this.logSplitter.Location = new System.Drawing.Point(0, 298);
			this.logSplitter.Name = "logSplitter";
			this.logSplitter.Size = new System.Drawing.Size(752, 3);
			this.logSplitter.TabIndex = 5;
			this.logSplitter.TabStop = false;
			// 
			// MainForm
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(752, 449);
			this.Controls.Add(this.logSplitter);
			this.Controls.Add(this.logBox);
			this.Controls.Add(this.statusBar);
			this.IsMdiContainer = true;
			this.Menu = this.mainMenu;
			this.Name = "MainForm";
			this.StartPosition = System.Windows.Forms.FormStartPosition.Manual;
			this.Text = "MonoScan";
			this.ResumeLayout(false);

		}
		#endregion		

		MonoComm.MessageHandlerDelegate monoCommMessageHandlerDelegate;

		XmlDocument xmlDocument;
		XmlNode xmlNode;

		bool constructed;

		int commPort;
		uint[] baudRates;

		// Full main menu. Typical scanner adds separate menu item next to Connection menu item.

		public MainMenu MainMenu 
		{
			get { return mainMenu; }
		}

		// Log messages

		public LogBox LogBox
		{
			get { return logBox; }
		}

		// Status bar messages

		public int PushStatusMessage(string message)
		{
			statusMessageArrayList.Add(message);

			if (statusBar.Text.Length != 0)
				statusBar.Text += " - ";
			statusBar.Text += message;

			return statusMessageArrayList.Count - 1;
		}

		public void PopStatusMessage(int id)
		{
			statusMessageArrayList.RemoveRange(id, statusMessageArrayList.Count - id);

			string s = "";
			for (int index = 0; index < id; ++ index)
			{
				if (index != 0)
					s += " - ";
				s += (string) statusMessageArrayList[index];
			}

			statusBar.Text = s;
		}
	
		ArrayList statusMessageArrayList;

		// Idle. Some long-term ECU functions don't allow idle messages

		public bool IdleEnabled
		{
			get { return idleTimer.Enabled; }

			set 
			{
				if (! connected)
					throw new InvalidOperationException();

				idleTimer.Enabled = value;
			}
		}

		public void HandleMonoCommException(string message, MonoComm.Exception exception)
		{
			string messageBoxText = null;

			if (message != null)
			{
				messageBoxText += message;
				messageBoxText += "\n\n";
			}

			messageBoxText += exception.Message;

			if (exception.Status == MonoComm.Status.ErrorConnect ||
				exception.Status == MonoComm.Status.ErrorConnectAdapter ||
				exception.Status == MonoComm.Status.ErrorConnectBlockIndex ||
				exception.Status == MonoComm.Status.ErrorConnectEcu ||
				exception.Status == MonoComm.Status.ErrorConnectEcuEcho)
			{
				Disconnect();	// before the message box

				messageBoxText += "\n\nConnection has been closed.";
			}

			MessageBox.Show(this, messageBoxText, "MonoScan", MessageBoxButtons.OK, 
				MessageBoxIcon.Error);
		}

		public void HandleException(string message, Exception exception)
		{
			string messageBoxText = null;

			if (message != null)
			{
				messageBoxText += message;
				messageBoxText += "\n\n";
			}

			messageBoxText += exception.Message;

			MessageBox.Show(this, messageBoxText, "MonoScan", MessageBoxButtons.OK, 
				MessageBoxIcon.Error);
		}

		void scannerMenuItem_Popup(object sender, System.EventArgs e)
		{
			scannerConnectMenuItem.Enabled = ! connected;
			scannerDisconnectMenuItem.Enabled = connected;
			scannerLogMenuItem.Checked = logBox.Visible;
		}

		void scannerConnectMenuItem_Click(object sender, System.EventArgs e)
		{
			try
			{
				string commPortName = "\\\\.\\Com" + commPort.ToString();

				MonoComm.Status status; string[] descriptions;

				if (baudRates.Length > 0)
				{
					// There're baud rate hints, trying them in the direct order

					int index = 0;

					do 
					{
						uint baudRate = baudRates[index];

						int statusMessageId = PushStatusMessage("Connecting expecting " + baudRate.ToString() + " baud rate");

						try
						{
							Update();

							// Connecting and getting the identification descriptions

							status = MonoComm.Connect(commPortName, baudRate, 1, out descriptions);
						}
						finally
						{
							PopStatusMessage(statusMessageId);
						}
					}
					while (status == MonoComm.Status.ErrorConnect && ++ index < baudRates.Length);
				}
				else
				{
					int statusMessageId = PushStatusMessage("Connecting");

					try
					{
						Update();

						// Connecting and getting the identification descriptions

						status = MonoComm.Connect(commPortName, 0, 1, out descriptions);
					}
					finally
					{
						PopStatusMessage(statusMessageId);
					}
				}

				if (status != MonoComm.Status.Ok)
					throw new MonoComm.Exception(status);

				try
				{
					// Creating the appropriate scanner to interact with the control unit

					scanner = CreateScanner(descriptions);

					try
					{
						connectedStatusMessageId = PushStatusMessage(GetConnectedStatusMessage(descriptions));

						connected = true;
						idleTimer.Enabled = true;
					}
					catch
					{
						scanner.Dispose();
						scanner = null;

						throw;
					}
				}
				catch
				{
					MonoComm.Disconnect();

					throw;
				}
			}
			catch (Exception exception)
			{
				MessageBox.Show(this, "Couldn't connect\n\n" + exception.Message, 
					"MonoScan", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}

		Scanner CreateScanner(string[] descriptions)
		{
			if (descriptions.Length == 0)
				throw new Exception("No control unit descriptions retrieved");

			Scanner scanner;

			XmlNode controlUnitXmlNode = Configuration.FindXmlNode(xmlNode, "ControlUnits/ControlUnit[@Code=\"{0}\"]",
				descriptions[0].Trim());			

			if (controlUnitXmlNode != null)
			{
				string scannerName = Configuration.GetXmlNodeAttribute(controlUnitXmlNode, "Scanner");

				// Creating appropriate scanner instance based on name

				switch (scannerName)
				{
					case "MonoMotronic121":
					{
						XmlNode scannerXmlNode = Configuration.FindXmlNode(xmlNode, "Scanners/Scanner[@Name=\"MonoMotronic121\"]");
						scanner = new MonoMotronic121.Scanner(this, scannerXmlNode);

						break;
					}
					default:
					{
						throw new Exception("Invalid scanner name '" + scannerName + "'");
					}
				}
			}
			else
			{
				// Unsupported control unit. We let the user see the descriptions 
				// in the status bar and keep the connection.					

				scanner = new Scanner(this, null);
			}

			return scanner;
		}

		string GetConnectedStatusMessage(string[] descriptions)
		{
			StringBuilder stringBuilder = new StringBuilder();

			stringBuilder.Append("[");

			for (int index = 0; index < descriptions.Length; ++ index)
			{
				if (index != 0)
					stringBuilder.Append(" | ");

				stringBuilder.Append(descriptions[index]);
			}

			stringBuilder.Append("]");

			return stringBuilder.ToString();
		}

		Scanner scanner;
		
		int connectedStatusMessageId;		
		bool connected;

		void idleTimer_Tick(object sender, System.EventArgs e)
		{
			MonoComm.Status status = MonoComm.Idle();

			if (status != MonoComm.Status.Ok)
			{
				HandleMonoCommException(null, new MonoComm.Exception(status));

				return;
			}
		}

		void scannerDisconnectMenuItem_Click(object sender, System.EventArgs e)
		{
			Disconnect();
		}

		void scannerOptionsMenuItem_Click(object sender, System.EventArgs e)
		{
			try
			{
				OptionsForm optionsForm = new OptionsForm();

				optionsForm.CommPort = commPort;
				optionsForm.BaudRates = baudRates;
				optionsForm.MessageLevel = MonoComm.GetMessageLevel();

				DialogResult dialogReasult = optionsForm.ShowDialog(this);

				if (dialogReasult == DialogResult.OK)
				{
					baudRates = optionsForm.BaudRates;
					commPort = optionsForm.CommPort;
					MonoComm.SetMessageLevel(optionsForm.MessageLevel);
				}
			}
			catch (Exception exception)
			{
				HandleException(null, exception);
			}
		}

		void scannerLogMenuItem_Click(object sender, System.EventArgs e)
		{
			if (logBox.Visible)
			{
				logBox.Visible = false;
				logSplitter.Visible = false;				
			}
			else
			{
				logSplitter.Visible = true;
				logBox.Visible = true;
			}
		}

		void WindowMenuItem_Popup(object sender, System.EventArgs e)
		{
			bool enabled = MdiChildren.Length != 0;

			for (int index = 0; index < windowMenuItem.MenuItems.Count; ++ index)
				windowMenuItem.MenuItems[index].Enabled = enabled;
		}

		void windowTileHorizontallyMenuItem_Click(object sender, System.EventArgs e)
		{
			LayoutMdi(MdiLayout.TileHorizontal);
		}

		void windowTileVerticallyMenuItem_Click(object sender, System.EventArgs e)
		{
			LayoutMdi(MdiLayout.TileVertical);
		}

		void windowCascadeMenuItem_Click(object sender, System.EventArgs e)
		{
			LayoutMdi(MdiLayout.Cascade);
		}

		void windowCloseAllMenuItem_Click(object sender, System.EventArgs e)
		{
			Form[] children = MdiChildren;

			for (int index = 0; index < children.Length; ++ index)
				children[index].Close();
		}

		void scannerExitMenuItem_Click(object sender, System.EventArgs e)
		{
			Close();
		}

		void helpAboutMenuItem_Click(object sender, System.EventArgs e)
		{
			AboutForm aboutForm = new AboutForm();
			aboutForm.ShowDialog();
			aboutForm.Dispose();
		}

		void MonoCommMessageHandler(IntPtr contextPtr, string message)
		{
			logBox.AppendText(message);
			logBox.Update();
		}

		protected override void OnClosing(CancelEventArgs e)
		{			
			e.Cancel = false;	// in some cases the value by default is true (!)
			base.OnClosing(e);
		}

		protected override void OnResize(EventArgs e)
		{
			if (WindowState == FormWindowState.Normal)
				normalSize = Size;

			base.OnResize (e);
		}

		Size normalSize;

		protected override void OnMove(EventArgs e)
		{
			if (WindowState == FormWindowState.Normal)
				normalLocation = Location;

			base.OnMove (e);
		}

		System.Drawing.Point normalLocation;

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			// Dispose can be called any time

			try
			{
				if (disposing)
				{
					if (connected)
					{
						scanner.Dispose();

						idleTimer.Enabled = false;
					}

					if (components != null) 
					{
						components.Dispose();
					}

					try
					{
						WriteConfiguration();

						XmlTextWriter xmlTextWriter = new XmlTextWriter(Configuration.Path, null);
						xmlTextWriter.IndentChar = '\t';
						xmlTextWriter.Indentation = 1;
						xmlTextWriter.Formatting = Formatting.Indented;						
						xmlDocument.Save(xmlTextWriter);
						xmlTextWriter.Close();						
					}
					catch
					{
					}
				}

				if (connected)
				{
					MonoComm.Disconnect();
					connected = false;
				}

				if (constructed)
				{
					MonoComm.Destruct();
					constructed = false;
				}
			}
			finally
			{
				base.Dispose(disposing);
			}
		}

		void Disconnect()
		{
			try
			{
				PushStatusMessage("Disconnecting");
			}
			catch
			{
			}

			try
			{
				Update();
			}
			catch 
			{
			}

			try
			{
				scanner.Dispose();
			}
			catch 
			{
			}
			
			scanner = null;			

			try
			{
				MonoComm.Disconnect();
			}
			catch
			{
			}

			try
			{
				idleTimer.Enabled = false;
			}
			catch
			{
			}

			try
			{
				PopStatusMessage(connectedStatusMessageId);
			}
			catch
			{
			}
		
			connected = false;
		}

		void ReadConfiguration()
		{
			XmlNode settingsXmlNode = Configuration.FindXmlNode(xmlNode, "Settings");

			if (settingsXmlNode != null)
			{
				commPort = int.Parse(Configuration.GetXmlNodeAttribute(settingsXmlNode, "CommPort"));

				XmlNode baudRatesXmlNode = Configuration.GetXmlNode(settingsXmlNode, "BaudRates");

				XmlNodeList baudRateXmlNodeList = Configuration.FindXmlNodes(baudRatesXmlNode, "BaudRate");
				baudRates = new uint[baudRateXmlNodeList.Count];
				for (int index = 0; index < baudRateXmlNodeList.Count; ++ index)
					baudRates[index] = uint.Parse(Configuration.GetXmlNodeAttribute(baudRateXmlNodeList[index], "Value"));

				string messagesLevel = Configuration.GetXmlNodeAttribute(settingsXmlNode, "MessagesLevel");
				MonoComm.SetMessageLevel((MonoComm.MessageLevel) Enum.Parse(typeof(MonoComm.MessageLevel), messagesLevel));
			}
			else
			{
				commPort = 1;
				baudRates = new uint[] {10400, 9600, 4800};
			}

			XmlNode windowXmlNode = Configuration.FindXmlNode(xmlNode, "Window");

			if (windowXmlNode != null)
			{
				SetBounds(int.Parse(Configuration.GetXmlNodeAttribute(windowXmlNode, "Left")),
					int.Parse(Configuration.GetXmlNodeAttribute(windowXmlNode, "Top")),
					int.Parse(Configuration.GetXmlNodeAttribute(windowXmlNode, "Width")),
					int.Parse(Configuration.GetXmlNodeAttribute(windowXmlNode, "Height")) - 19 /* ? */);
			}

			XmlNode loggingXmlNode = Configuration.FindXmlNode(xmlNode, "Logging");

			if (loggingXmlNode != null)
			{
				logBox.Height = int.Parse(Configuration.GetXmlNodeAttribute(loggingXmlNode, "Height"));

				bool enabled = bool.Parse(Configuration.GetXmlNodeAttribute(loggingXmlNode, "Enabled"));

				if (enabled)
				{
					logSplitter.Visible = true;
					logBox.Visible = true;
				}
				else
				{			
					logBox.Visible = false;
					logSplitter.Visible = false;
				}
			}
			else
			{
				logBox.Visible = false;
				logSplitter.Visible = false;
			}
		}

		void WriteConfiguration()
		{
			XmlNode settingsXmlNode = Configuration.CreateChildXmlNode(xmlNode, "Settings");
			Configuration.CreateXmlNodeAttribute(settingsXmlNode, "CommPort", commPort.ToString());

			XmlNode baudRatesXmlNode = Configuration.CreateChildXmlNode(settingsXmlNode, "BaudRates");
			for (int index = 0; index < baudRates.Length; ++ index)
			{
				XmlNode baudRateXmlNode = Configuration.CreateChildXmlNodes(baudRatesXmlNode, "BaudRate");
				Configuration.CreateXmlNodeAttribute(baudRateXmlNode, "Value", baudRates[index].ToString());
			}

			Configuration.CreateXmlNodeAttribute(settingsXmlNode, "MessagesLevel", MonoComm.GetMessageLevel().ToString());

			XmlNode windowXmlNode = Configuration.CreateChildXmlNode(xmlNode, "Window");
			Configuration.CreateXmlNodeAttribute(windowXmlNode, "Left", normalLocation.X.ToString());
			Configuration.CreateXmlNodeAttribute(windowXmlNode, "Top", normalLocation.Y.ToString());
			Configuration.CreateXmlNodeAttribute(windowXmlNode, "Width", normalSize.Width.ToString());
			Configuration.CreateXmlNodeAttribute(windowXmlNode, "Height", normalSize.Height.ToString());

			XmlNode loggingXmlNode = Configuration.CreateChildXmlNode(xmlNode, "Logging");
			Configuration.CreateXmlNodeAttribute(loggingXmlNode, "Height", logBox.Height.ToString());
			Configuration.CreateXmlNodeAttribute(loggingXmlNode, "Enabled", logBox.Visible.ToString());
		}
	}
}
